import type { PlaywrightTestConfig } from '@playwright/test';
import { devices, expect } from '@playwright/test';

const projects: PlaywrightTestConfig['projects'] = [
  {
    /**
     * This is really just desktop Firefox
     * but with a mobile viewport.
     */
    name: 'Mobile Firefox',
    use: {
      browserName: 'firefox',
      /**
       * This is the Pixel 5 configuration.
       * We can't use devices['Pixel 5']
       * because the "isMobile" option is
       * not supported on Firefox.
       */
      viewport: {
        width: 393,
        height: 727
      }
    },
  },
  {
    name: 'Mobile Chrome',
    use: {
      browserName: 'chromium',
      ...devices['Pixel 5']
    }
  },
  {
    name: 'Mobile Safari',
    use: {
      browserName: 'webkit',
      ...devices['iPhone 12']
    }
  }
];

/**
 * See https://playwright.dev/docs/test-configuration.
 */
const config: PlaywrightTestConfig = {
  testMatch: '*.e2e.ts',
  expect: {
    /**
     * Maximum time expect() should wait for the condition to be met.
     * For example in `await expect(locator).toHaveText();`
     */
    timeout: 5000,
    
    /**
     * Configure screenshot comparison settings
     * to be more tolerant of minor visual differences
     */
    toHaveScreenshot: {
      // Increase the threshold to allow for small font rendering differences
      // This sets the maximum allowed ratio of pixels that can be different
      maxDiffPixelRatio: 0.02, // Allow up to 2% of pixels to be different
      
      // Alternatively, you can use absolute pixel count
      // maxDiffPixels: 100, // Allow up to 100 pixels to be different
      
      // Add a threshold for per-pixel difference to handle anti-aliasing
      threshold: 0.2, // Default is 0.1
    }
  },
  /* Fail the build on CI if you accidentally left test.only in the source code. */
  forbidOnly: !!process.env.CI,
  retries: process.env.CI ? 2 : 0,
  /* Opt out of parallel tests on CI. */
  workers: process.env.CI ? 1 : undefined,
  /* Reporter to use. See https://playwright.dev/docs/test-reporters */
  reporter: 'html',
  /* Shared settings for all the projects below. See https://playwright.dev/docs/api/class-testoptions. */
  use: {
    /* Maximum time each action such as `click()` can take. Defaults to 0 (no limit). */
    actionTimeout: 0,
    /* Base URL to use in actions like `await page.goto('/')`. */
    // baseURL: 'http://localhost:3000',

    /**
     * All failed tests should create
     * a trace file for easier debugging.
     *
     * See https://playwright.dev/docs/trace-viewer
     */
    trace: 'retain-on-failure',
    baseURL: 'http://localhost:3333',
  },

  /* Configure projects for major browsers */
  projects,
  webServer: {
    command: 'serve www -p 3333',
    port: 3333,
    reuseExistingServer: !process.env.CI
  }
};

export default config;