/**
 * Copyright IBM Corp. 2020, 2023
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { render } from 'lit';
import BXStructuredListRow from '../../src/components/structured-list/structured-list-row';
import { Playground } from '../../src/components/structured-list/structured-list.stories';

const template = (props?) =>
  Playground({
    'cds-structured-list': props,
  });

xdescribe('cds-structured-list', () => {
  describe('Selection', () => {
    let list;
    let rows;

    beforeEach(async () => {
      render(template({ hasSelection: true }), document.body);
      await Promise.resolve();
      list = document.body.querySelector('cds-structured-list');
      rows = document.body.querySelectorAll('cds-structured-list-row');
    });

    it('should reflect the selection settings', async () => {
      list.setAttribute('selection-name', 'selection-name-foo');
      await Promise.resolve();
      expect(
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
        list
          .shadowRoot!.querySelector('section')!
          .classList.contains('cds--structured-list--selection')
      ).toBe(true);
      expect(
        Array.prototype.every.call(
          rows,
          (row) => row.selectionName === 'selection-name-foo'
        )
      ).toBe(true);
    });

    it('should reflect the selection', () => {
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
      const input1 = rows[1]!.shadowRoot!.querySelector('input');
      // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
      input1!.click();
      expect(
        Array.prototype.map.call(
          rows,
          (item) => (item as BXStructuredListRow).selected
        )
      ).toEqual([false, true, false]);
    });

    it('Should navigate by up/down keys', () => {
      (rows[0] as HTMLElement).focus();
      const event = new CustomEvent('keydown', {
        bubbles: true,
        composed: true,
      });
      rows[0].dispatchEvent(Object.assign(event, { key: 'ArrowDown' }));
      expect(
        Array.prototype.map.call(
          rows,
          (item) => (item as BXStructuredListRow).selected
        )
      ).toEqual([false, true, false]);
      expect(Array.prototype.map.call(rows, (row) => row.tabIndex)).toEqual([
        -1, 0, -1,
      ]);
      rows[1].dispatchEvent(Object.assign(event, { key: 'ArrowUp' }));
      expect(
        Array.prototype.map.call(
          rows,
          (item) => (item as BXStructuredListRow).selected
        )
      ).toEqual([true, false, false]);
      expect(Array.prototype.map.call(rows, (row) => row.tabIndex)).toEqual([
        0, -1, -1,
      ]);
    });
  });

  afterEach(async () => {
    // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- https://github.com/carbon-design-system/carbon/issues/20452
    await render(undefined!, document.body);
  });
});
