/**
 * Copyright IBM Corp. 2019, 2025
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { classMap } from 'lit/directives/class-map.js';
import { html } from 'lit';
import { property, query } from 'lit/decorators.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { prefix } from '../../globals/settings';
import CDSCheckbox from '../checkbox/checkbox';
import { TOGGLE_SIZE } from './defs';
import styles from './toggle.scss?lit';
import HostListenerMixin from '../../globals/mixins/host-listener';
import { carbonElement as customElement } from '../../globals/decorators/carbon-element';

export { TOGGLE_SIZE };

/**
 * Basic toggle.
 *
 * @element cds-toggle
 * @slot label-text - The label text.
 * @slot checked-text - The text for the checked state.
 * @slot unchecked-text - The text for the unchecked state.
 * @fires cds-toggle-changed - The custom event fired after this changebox changes its checked state.
 */
@customElement(`${prefix}-toggle`)
class CDSToggle extends HostListenerMixin(CDSCheckbox) {
  @query('button')
  protected _checkboxNode!: HTMLInputElement;

  /**
   * Handles `click` event on the `<button>` in the shadow DOM.
   */
  protected _handleChange() {
    const { checked } = this._checkboxNode;
    if (this.disabled || this.readOnly) {
      return;
    }
    this.toggled = !checked;
    const { eventChange } = this.constructor as typeof CDSToggle;
    this.dispatchEvent(
      new CustomEvent(eventChange, {
        bubbles: true,
        composed: true,
        detail: {
          checked: this.toggled, // TODO: remove in v12
          toggled: this.toggled,
        },
      })
    );
  }

  protected _renderCheckmark() {
    if (this.size !== TOGGLE_SIZE.SMALL || this.readOnly == true) {
      return undefined;
    }
    return html`
      <svg
        class="${prefix}--toggle__check"
        width="6px"
        height="5px"
        viewBox="0 0 6 5">
        <path d="M2.2 2.7L5 0 6 1 2.2 5 0 2.7 1 1.5z" />
      </svg>
    `;
  }

  // TODO: remove in v12
  /**
   * @deprecated Use `toggled` instead.
   * The `checked` attribute will be removed in the next major version.
   */
  declare checked: boolean;
  // TODO: remove in v12
  /**
   *
   * **Deprecated:** Use `toggled` instead.
   * The `checked` attribute will be removed in the next major version.
   */
  @property({ type: Boolean, attribute: 'checked', reflect: true })
  get _checkedAttributeAlias() {
    return this.toggled;
  }
  set _checkedAttributeAlias(v: boolean) {
    this.toggled = v;
  }

  // TODO: remove get() and set() in v12
  /**
   * Specify whether the control is toggled
   */
  @property({ type: Boolean, reflect: true })
  get toggled(): boolean {
    return this.checked;
  }
  set toggled(v: boolean) {
    const prev = this.checked;
    const next = v;
    if (prev === next) return;
    this.checked = v;

    this.requestUpdate('toggled', prev);
    this.requestUpdate('_checkedAttributeAlias');
  }
  /**
   * Specify another element's id to be used as the label for this toggle
   */
  @property({ type: String, attribute: 'aria-labelledby' })
  ariaLabelledby?: string;

  // TODO: swap value with labelB in v12 to match React
  /**
   * Specify the label for the "on" position
   */
  @property({ attribute: 'label-a' })
  labelA = 'On';

  /**
   * Hide label text.
   */
  @property({ reflect: true, type: Boolean })
  hideLabel = false;

  /**
   * Read only boolean.
   */
  @property({ reflect: true, attribute: 'read-only', type: Boolean })
  readOnly = false;

  /**
   * Toggle size.
   */
  @property({ reflect: true })
  size = TOGGLE_SIZE.REGULAR;

  // TODO: swap value with labelA in v12 to match React
  /**
   * Specify the label for the "off" position
   */
  @property({ attribute: 'label-b' })
  labelB = 'Off';

  /**
   * Private references of external <label> elements that are
   * `for="this-toggle-element-id"`
   */
  private _externalLabels: HTMLLabelElement[] = [];

  /**
   * Handles `click` on external `<label>`
   */
  private _onExternalLabelClick = () => {
    this._checkboxNode?.focus();
    this._handleChange();
  };

  /**
   * Finds external toggle `<label>`s and attaches handlers.
   */
  private _attachExternalLabels() {
    const doc = this.ownerDocument || document;

    const found = this.id
      ? [...doc.querySelectorAll<HTMLLabelElement>(`label[for="${this.id}"]`)]
      : [];

    this._externalLabels = Array.from(new Set(found));
    this._externalLabels.forEach((lbl) => {
      lbl.addEventListener('click', this._onExternalLabelClick);
    });
  }

  connectedCallback() {
    super.connectedCallback?.();
    this._attachExternalLabels();
  }

  disconnectedCallback() {
    super.disconnectedCallback?.();
    this._externalLabels.forEach((lbl) =>
      lbl.removeEventListener('click', this._onExternalLabelClick)
    );
  }

  render() {
    const {
      toggled,
      disabled,
      labelText,
      hideLabel,
      id,
      name,
      size,
      labelA,
      labelB,
      value,
      _handleChange: handleChange,
    } = this;
    const inputClasses = classMap({
      [`${prefix}--toggle__appearance`]: true,
      [`${prefix}--toggle__appearance--${size}`]: size,
    });
    const toggleClasses = classMap({
      [`${prefix}--toggle__switch`]: true,
      [`${prefix}--toggle__switch--checked`]: toggled,
    });

    const labelTextClasses = classMap({
      [`${prefix}--toggle__label-text`]: labelText,
      [`${prefix}--visually-hidden`]: hideLabel,
    });

    let stateText = '';

    if (hideLabel) {
      stateText = labelText || '';
    } else {
      stateText = toggled ? labelA : labelB;
    }

    const labelId = id ? `${id}_label` : undefined;

    const hasLabelText = (this.labelText ?? '') !== '';

    const ariaLabelledby = this.ariaLabelledby ?? (hasLabelText && labelId);

    return html`
      <button
        class="${prefix}--toggle__button"
        role="switch"
        type="button"
        aria-checked=${toggled}
        aria-labelledby=${ifDefined(ariaLabelledby)}
        .checked=${toggled}
        name="${ifDefined(name)}"
        value="${ifDefined(value)}"
        ?disabled=${disabled}
        id="${id}"
        @click=${handleChange}></button>
      <label for="${id}" class="${prefix}--toggle__label">
        ${labelText
          ? html`<span class="${labelTextClasses}">${labelText}</span>`
          : null}
        <div class="${inputClasses}">
          <div class="${toggleClasses}">${this._renderCheckmark()}</div>
          <span class="${prefix}--toggle__text" aria-hidden="true"
            >${stateText}</span
          >
        </div>
      </label>
    `;
  }

  /**
   * The name of the custom event fired after this changebox changes its toggled state.
   */
  static get eventChange() {
    return `${prefix}-toggle-changed`;
  }

  static styles = styles;
}

export default CDSToggle;
