/**
 * Copyright IBM Corp. 2019, 2025
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { html } from 'lit';
import { ifDefined } from 'lit/directives/if-defined.js';
import { iconLoader } from '../../globals/internal/icon-loader';
import View16 from '@carbon/icons/es/view/16.js';
import FolderOpen16 from '@carbon/icons/es/folder--open/16.js';
import Folders16 from '@carbon/icons/es/folders/16.js';
import { INPUT_SIZE } from '../text-input/text-input';
import './number-input';
import './number-input-skeleton';
import '../form/form-item';
import '../ai-label';
import '../icon-button';

const content = html`
  <div slot="body-text">
    <p class="secondary">AI Explained</p>
    <h2 class="ai-label-heading">84%</h2>
    <p class="secondary bold">Confidence score</p>
    <p class="secondary">
      Lorem ipsum dolor sit amet, di os consectetur adipiscing elit, sed do
      eiusmod tempor incididunt ut fsil labore et dolore magna aliqua.
    </p>
    <hr />
    <p class="secondary">Model type</p>
    <p class="bold">Foundation model</p>
  </div>
`;

const actions = html`
  <cds-icon-button kind="ghost" slot="actions" size="lg">
    ${iconLoader(View16, { slot: 'icon' })}
    <span slot="tooltip-content"> View </span>
  </cds-icon-button>
  <cds-icon-button kind="ghost" slot="actions" size="lg">
    ${iconLoader(FolderOpen16, { slot: 'icon' })}
    <span slot="tooltip-content"> Open folder</span>
  </cds-icon-button>
  <cds-icon-button kind="ghost" slot="actions" size="lg">
    ${iconLoader(Folders16, { slot: 'icon' })}
    <span slot="tooltip-content"> Folders </span>
  </cds-icon-button>
  <cds-ai-label-action-button>View details</cds-ai-label-action-button>
`;

const sizes = {
  [`Small size (${INPUT_SIZE.SMALL})`]: INPUT_SIZE.SMALL,
  [`Medium size (${INPUT_SIZE.MEDIUM})`]: INPUT_SIZE.MEDIUM,
  [`Large size (${INPUT_SIZE.LARGE})`]: INPUT_SIZE.LARGE,
};

const args = {
  allowEmpty: false,
  decrementButtonDescription: 'decrease number input',
  incrementButtonDescription: 'increase number input',
  disabled: false,
  helperText: 'Optional helper text',
  hideLabel: false,
  hideSteppers: false,
  invalid: false,
  invalidText: 'Number is not valid',
  label: 'number-input label',
  readOnly: false,
  value: 50,
  warn: false,
  warnText: 'Warning text',
  min: 0,
  max: 100,
  step: 1,
  size: INPUT_SIZE.MEDIUM,
};

const argTypes = {
  allowEmpty: {
    control: 'boolean',
    description: '<code>true</code> to allow empty string.',
  },
  decrementButtonDescription: {
    control: 'text',
    description:
      'Decrement button assistive description (decrement-button-assistive-text)',
  },
  incrementButtonDescription: {
    control: 'text',
    description:
      'Increment button assistive description (increment-button-assistive-text)',
  },
  disabled: {
    control: 'boolean',
    description: 'Specify if the control should be disabled, or not.',
  },
  helperText: {
    control: 'text',
    description:
      'Provide text that is used alongside the control label for additional help.',
  },
  hideLabel: {
    control: 'boolean',
    description:
      'Specify whether you want the underlying label to be visually hidden.',
  },
  hideSteppers: {
    control: 'boolean',
    description: 'Specify whether you want the steppers to be hidden.',
  },
  invalid: {
    control: 'boolean',
    description: 'Specify if the currently value is invalid.',
  },
  invalidText: {
    control: 'text',
    description: 'Message which is displayed if the value is invalid.',
  },
  label: {
    control: 'text',
    description:
      'Generic <code>label</code> that will be used as the textual representation of what this field is for.',
  },
  readOnly: {
    control: 'boolean',
    description: 'Specify if the component should be read-only.',
  },
  value: {
    control: 'number',
    description: 'Specify the value of the input.',
  },
  warn: {
    control: 'boolean',
    description: 'Specify whether the control is currently in warning state.',
  },
  warnText: {
    control: 'text',
    description:
      'Provide the text that is displayed when the control is in warning state.',
  },
  min: {
    control: 'number',
    description: 'The minimum value.',
  },
  max: {
    control: 'number',
    description: 'The maximum value.',
  },
  step: {
    control: 'number',
    description:
      'Specify how much the values should increase/decrease upon clicking on up/down button.',
  },
  size: {
    control: 'select',
    description: 'Specify the size of the Number Input.',
    options: sizes,
  },
  onInput: {
    action: 'input',
  },
};

export const Default = {
  render: () => {
    return html`
      <cds-form-item>
        <cds-number-input
          value="50"
          min="0"
          max="100"
          size="${ifDefined(INPUT_SIZE.MEDIUM)}"
          step="1"
          label="number-input label"
          helper-text="Optional helper text">
        </cds-number-input>
      </cds-form-item>
    `;
  },
};

export const Skeleton = {
  parameters: {
    percy: {
      skip: true,
    },
  },
  render: () => html` <cds-number-input-skeleton></cds-number-input-skeleton> `,
};

export const WithAILabel = {
  render: () => html`
    <div style="width: 400px">
      <cds-number-input
        value="50"
        min="0"
        max="100"
        step="1"
        label="Number input"
        helper-text="Optional helper text.">
        <cds-ai-label alignment="bottom-left">
          ${content}${actions}
        </cds-ai-label>
      </cds-number-input>
    </div>
  `,
};

export const Playground = {
  args,
  argTypes,
  render: (args) => {
    const {
      allowEmpty,
      decrementButtonDescription,
      incrementButtonDescription,
      disabled,
      helperText,
      hideLabel,
      hideSteppers,
      invalid,
      invalidText,
      label,
      readOnly,
      warn,
      warnText,
      value,
      min,
      max,
      size,
      step,
      onInput,
    } = args ?? {};
    return html`
      <cds-form-item>
        <cds-number-input
          ?allow-empty="${allowEmpty}"
          decrement-button-assistive-text="${ifDefined(
            decrementButtonDescription
          )}"
          increment-button-assistive-text="${ifDefined(
            incrementButtonDescription
          )}"
          helper-text="${ifDefined(helperText)}"
          ?hide-steppers="${hideSteppers}"
          ?hide-label="${hideLabel}"
          ?invalid="${invalid}"
          invalid-text="${ifDefined(invalidText)}"
          label="${ifDefined(label)}"
          ?readonly="${readOnly}"
          value="${ifDefined(value)}"
          ?warn="${warn}"
          warn-text="${ifDefined(warnText)}"
          ?disabled="${disabled}"
          min="${ifDefined(min)}"
          max="${ifDefined(max)}"
          size="${ifDefined(size)}"
          step="${ifDefined(step)}"
          @input="${onInput}">
        </cds-number-input>
      </cds-form-item>
    `;
  },
};

// Hidden Test-Only Story for an issue where invalid with AI-Label had incorrect styling. #20117
export const InvalidWithAILabel = {
  tags: ['!dev', '!autodocs'], // hide story

  render: () => html`
    <div style="width: 400px">
      <cds-number-input
        value="50"
        min="0"
        max="100"
        step="1"
        label="Number input"
        invalid
        invalid-text="invalid">
        <cds-ai-label alignment="bottom-left">
          ${content}${actions}
        </cds-ai-label>
      </cds-number-input>
    </div>
  `,
};

// Hidden Test-Only Story for an issue where warn with AI-Label had incorrect styling. #20117
export const WarnWithAILabel = {
  tags: ['!dev', '!autodocs'], // hide story

  render: () => html`
    <div style="width: 400px">
      <cds-number-input
        value="50"
        min="0"
        max="100"
        step="1"
        label="Number input"
        warn
        warn-text="warning">
        <cds-ai-label alignment="bottom-left">
          ${content}${actions}
        </cds-ai-label>
      </cds-number-input>
    </div>
  `,
};

// Hidden Test-Only Story for an issue where disabled with AI-Label had incorrect styling. #20117
export const DisabledWithAILabel = {
  tags: ['!dev', '!autodocs'], // hide story

  render: () => html`
    <div style="width: 400px">
      <cds-number-input
        value="50"
        min="0"
        max="100"
        step="1"
        label="Number input"
        helper-text="Optional helper text."
        disabled>
        <cds-ai-label alignment="bottom-left">
          ${content}${actions}
        </cds-ai-label>
      </cds-number-input>
    </div>
  `,
};

export default {
  title: 'Components/Number Input',
};
