/**
 * Copyright IBM Corp. 2019, 2026
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { LitElement, html } from 'lit';
import { property } from 'lit/decorators.js';
import { prefix } from '../../globals/settings';
import Close20 from '@carbon/icons/es/close/20.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import FocusMixin from '../../globals/mixins/focus';
import { iconLoader } from '../../globals/internal/icon-loader';
import styles from './modal.scss?lit';
import { carbonElement as customElement } from '../../globals/decorators/carbon-element';
import '../icon-button';

/**
 * Modal close button.
 *
 * @element cds-modal-close-button
 * @csspart button The button.
 * @csspart close-icon The close icon.
 */
@customElement(`${prefix}-modal-close-button`)
class CDSModalCloseButton extends FocusMixin(LitElement) {
  /**
   * Specify a label for the close button of the modal; defaults to close
   */
  @property({ attribute: 'close-button-label' })
  closeButtonLabel = 'Close';

  updated() {
    this.parentElement?.setAttribute('close-button', '');
  }

  render() {
    const { closeButtonLabel } = this;
    return html`
      <cds-icon-button
        part="button"
        align="left"
        enter-delay-ms=""
        aria-label="${ifDefined(closeButtonLabel)}"
        kind="ghost"
        size="lg"
        leave-delay-ms="">
        ${iconLoader(Close20, {
          slot: 'icon',
          part: 'close-icon',
          class: `${prefix}--modal-close__icon`,
        })}
        <span slot="tooltip-content">${closeButtonLabel}</span>
      </cds-icon-button>
    `;
  }

  static shadowRootOptions = {
    ...LitElement.shadowRootOptions,
    delegatesFocus: true,
  };
  static styles = styles; // `styles` here is a `CSSResult` generated by custom WebPack loader
}

export default CDSModalCloseButton;
