/**
 * Copyright IBM Corp. 2016, 2026
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import { CaretDown } from '@carbon/icons-react';
import classNames from 'classnames';
import PropTypes from 'prop-types';
import React, {
  useEffect,
  useRef,
  useState,
  useCallback,
  useContext,
  type ElementType,
  type FocusEvent,
  type MouseEvent,
} from 'react';
import { keys, match, matches } from '../../internal/keyboard';
import { deprecate } from '../../prop-types/deprecate';
import { useControllableState } from '../../internal/useControllableState';
import { usePrefix } from '../../internal/usePrefix';
import { useId } from '../../internal/useId';
import { useFeatureFlag } from '../FeatureFlags';
import { IconButton } from '../IconButton';
import { TreeContext, DepthContext } from './TreeContext';

type UncontrolledOnToggle = (
  event: React.MouseEvent | React.KeyboardEvent,
  node: Pick<TreeNodeProps, 'id' | 'label' | 'value' | 'isExpanded'>
) => void;

type ControlledOnToggle = (isExpanded: TreeNodeProps['isExpanded']) => void;

export type TreeNodeProps = {
  /**
   * **Note:** this is controlled by the parent TreeView component, do not set manually.
   * The ID of the active node in the tree
   *
   * @deprecated The `active` prop for `TreeNode` has
   * been deprecated after the introduction of context. It will be removed in the next major release.
   */
  active?: string | number;
  /**
   * Specify the children of the TreeNode
   */
  children?: React.ReactNode;
  /**
   * Specify an optional className to be applied to the TreeNode
   */
  className?: string;
  /**
   * **[Experimental]** The default expansion state of the node.
   * *This is only supported with the `enable-treeview-controllable` feature flag!*
   */
  defaultIsExpanded?: boolean;
  /**
   * **Note:** this is controlled by the parent TreeView component, do not set manually.
   * TreeNode depth to determine spacing
   *
   * @deprecated The `depth` prop for `TreeNode` has
   * been deprecated after the introduction of context. It will be removed in the next major release.
   */
  depth?: number;
  /**
   * Specify if the TreeNode is disabled
   */
  disabled?: boolean;
  /**
   * Specify the TreeNode's ID. Must be unique in the DOM and is used for props.active, props.selected and aria-owns
   */
  id?: string;
  /**
   * Specify if the TreeNode is expanded (only applicable to parent nodes)
   */
  isExpanded?: boolean;
  /**
   * Rendered label for the TreeNode
   */
  label: React.ReactNode;
  /**
   * Callback function for when the node receives or loses focus
   *
   * @deprecated The `onNodeFocusEvent` prop for `TreeNode` has
   * been deprecated after the introduction of context. It will be removed in the next major release.
   */
  onNodeFocusEvent?: (event: React.FocusEvent<HTMLElement>) => void;
  /**
   * Callback function for when the node is selected
   */
  onSelect?: (
    event: React.MouseEvent | React.KeyboardEvent,
    node: Pick<TreeNodeProps, 'id' | 'label' | 'value'>
  ) => void;
  /**
   * Callback function for when a parent node is expanded or collapsed
   */
  onToggle?: UncontrolledOnToggle | ControlledOnToggle;
  /**
   * Callback function for when any node in the tree is selected
   *
   * @deprecated The `onTreeSelect` prop for `TreeNode` has
   * been deprecated after the introduction of context. It will be removed in the next major release.
   */
  onTreeSelect?: (
    event: React.MouseEvent | React.KeyboardEvent,
    node: Pick<TreeNodeProps, 'id' | 'label' | 'value'>
  ) => void;
  /**
   * A component used to render an icon.
   */
  renderIcon?: ElementType;
  /**
   * **Note:** this is controlled by the parent TreeView component, do not set manually.
   * Array containing all selected node IDs in the tree
   * @deprecated The `selected` prop for `TreeNode` has
   * been deprecated after the introduction of context. It will be removed in the next major release.
   */
  selected?: Array<string | number>;
  /**
   * Specify the value of the TreeNode
   */
  value?: string;
  /**
   * Optional: The URL the TreeNode is linking to
   */
  href?: string;
  /**
   *
   * Specify how the trigger should align with the tooltip when text is truncated
   */

  align?:
    | 'top'
    | 'bottom'
    | 'left'
    | 'right'
    | 'top-start'
    | 'top-end'
    | 'bottom-start'
    | 'bottom-end'
    | 'left-end'
    | 'left-start'
    | 'right-end'
    | 'right-start';

  /**
   * **Experimental**: Will attempt to automatically align the floating
   * element to avoid collisions with the viewport and being clipped by
   * ancestor elements. Requires React v17+
   * @see https://github.com/carbon-design-system/carbon/issues/18714
   */
  autoAlign?: boolean;
} & Omit<React.LiHTMLAttributes<HTMLElement>, 'onSelect'>;

const extractTextContent = (node: React.ReactNode): string => {
  if (node === null || node === undefined) return '';
  if (typeof node === 'string') return node;
  if (typeof node === 'number') return String(node);
  if (typeof node === 'boolean') return String(node);

  if (Array.isArray(node)) {
    return node.map(extractTextContent).join('');
  }

  if (React.isValidElement(node)) {
    const element = node as React.ReactElement<{ children?: React.ReactNode }>;
    const children = element.props.children;
    return extractTextContent(children);
  }

  return '';
};

type HTMLElementOrAnchor = HTMLElement | HTMLAnchorElement | null;

const useEllipsisCheck = (
  label: React.ReactNode,
  detailsWrapperRef: React.RefObject<HTMLElementOrAnchor>
) => {
  const [isEllipsisApplied, setIsEllipsisApplied] = useState(false);
  const labelTextRef = useRef<HTMLSpanElement>(null);

  const checkEllipsis = useCallback(() => {
    const element = labelTextRef.current;
    if (!element) {
      setIsEllipsisApplied(false);
      return;
    }
    if (element.offsetWidth === 0) {
      setIsEllipsisApplied(false);
      return;
    }
    const checkElement = detailsWrapperRef.current || element;

    if (checkElement && checkElement.offsetWidth > 0) {
      const isTextTruncated = element.scrollWidth > checkElement.offsetWidth;
      setIsEllipsisApplied(isTextTruncated);
    } else {
      setIsEllipsisApplied(false);
    }
  }, [detailsWrapperRef]);

  useEffect(() => {
    const animationFrameId: number = requestAnimationFrame(checkEllipsis);

    let resizeObserver: ResizeObserver | undefined;
    if (
      typeof window !== 'undefined' &&
      typeof window.ResizeObserver !== 'undefined' &&
      labelTextRef.current
    ) {
      resizeObserver = new window.ResizeObserver(() => {
        requestAnimationFrame(checkEllipsis);
      });
      resizeObserver.observe(labelTextRef.current);

      if (detailsWrapperRef.current) {
        resizeObserver.observe(detailsWrapperRef.current);
      }
    }

    return () => {
      cancelAnimationFrame(animationFrameId);
      if (resizeObserver) {
        if (labelTextRef.current) {
          // eslint-disable-next-line  react-hooks/exhaustive-deps -- https://github.com/carbon-design-system/carbon/issues/20452
          resizeObserver.unobserve(labelTextRef.current);
        }
        if (detailsWrapperRef.current) {
          // eslint-disable-next-line  react-hooks/exhaustive-deps -- https://github.com/carbon-design-system/carbon/issues/20452
          resizeObserver.unobserve(detailsWrapperRef.current);
        }
        resizeObserver.disconnect();
      }
    };
  }, [checkEllipsis, detailsWrapperRef]);

  return {
    labelTextRef,
    isEllipsisApplied,
    tooltipText: extractTextContent(label),
  };
};

const TreeNode = React.forwardRef<HTMLElement, TreeNodeProps>(
  (
    {
      children,
      className,
      disabled,
      id: nodeId,
      isExpanded,
      defaultIsExpanded,
      label,
      onSelect: onNodeSelect,
      onToggle,
      renderIcon: Icon,
      value,
      href,
      align = 'bottom',
      autoAlign = false,
      // These props are fallback props if the TreeContext is not available or only TreeNode is used as a standalone component
      active: propActive,
      depth: propDepth,
      selected: propSelected,
      onTreeSelect: propOnTreeSelect,
      onNodeFocusEvent,
      ...rest
    },
    forwardedRef
  ) => {
    const treeContext = useContext(TreeContext);
    const contextDepth = useContext(DepthContext);

    // Prioritize direct props, and fall back to context values.
    const depth = propDepth ?? (contextDepth !== -1 ? contextDepth : 0);
    const active = propActive ?? treeContext?.active;
    const selected = propSelected ?? treeContext?.selected ?? [];
    const onTreeSelect = propOnTreeSelect ?? treeContext?.onTreeSelect;

    const detailsWrapperRef = useRef<HTMLElementOrAnchor>(null);
    const { labelTextRef, isEllipsisApplied, tooltipText } = useEllipsisCheck(
      label,
      detailsWrapperRef
    );

    const enableTreeviewControllable = useFeatureFlag(
      'enable-treeview-controllable'
    );

    // eslint-disable-next-line  react-hooks/rules-of-hooks -- https://github.com/carbon-design-system/carbon/issues/20452
    const { current: id } = useRef(nodeId || useId());

    const controllableExpandedState = useControllableState({
      value: isExpanded,
      onChange: onToggle as ControlledOnToggle,
      defaultValue: defaultIsExpanded ?? false,
    });
    const uncontrollableExpandedState = useState(isExpanded ?? false);
    const [expanded, setExpanded] = enableTreeviewControllable
      ? controllableExpandedState
      : uncontrollableExpandedState;

    const currentNode = useRef<HTMLElement | null>(null);
    const currentNodeLabel = useRef<HTMLDivElement>(null);
    const prefix = usePrefix();

    const nodeLabelId = `${id}__label`;

    const renderLabelText = () => {
      if (isEllipsisApplied && tooltipText) {
        return (
          <IconButton
            label={tooltipText}
            kind="ghost"
            align={align}
            autoAlign={autoAlign}
            className={`${prefix}--tree-node__label__text-button`}
            wrapperClasses={`${prefix}--popover-container`}>
            <span
              id={nodeLabelId}
              ref={labelTextRef}
              className={`${prefix}--tree-node__label__text`}>
              {label}
            </span>
          </IconButton>
        );
      }

      return (
        <span
          id={nodeLabelId}
          ref={labelTextRef}
          className={`${prefix}--tree-node__label__text`}>
          {label}
        </span>
      );
    };

    const setRefs = (element: HTMLElement | null) => {
      currentNode.current = element;
      if (typeof forwardedRef === 'function') {
        forwardedRef(element);
      } else if (forwardedRef) {
        forwardedRef.current = element;
      }
    };

    const isActive = active === id;
    const isSelected = selected?.includes(id) ?? false;

    const treeNodeClasses = classNames(className, `${prefix}--tree-node`, {
      [`${prefix}--tree-node--active`]: isActive,
      [`${prefix}--tree-node--disabled`]: disabled,
      [`${prefix}--tree-node--selected`]: isSelected,
      [`${prefix}--tree-node--with-icon`]: Icon,
      [`${prefix}--tree-leaf-node`]: !children,
      [`${prefix}--tree-parent-node`]: children,
    });
    const toggleClasses = classNames(
      `${prefix}--tree-parent-node__toggle-icon`,
      {
        [`${prefix}--tree-parent-node__toggle-icon--expanded`]: expanded,
      }
    );
    function handleToggleClick(event: React.MouseEvent<HTMLSpanElement>) {
      if (disabled) {
        return;
      }

      // Prevent the node from being selected
      event.stopPropagation();
      if (href) {
        event.preventDefault();
      }

      if (!enableTreeviewControllable) {
        (onToggle as UncontrolledOnToggle)?.(event, {
          id,
          isExpanded: !expanded,
          label,
          value,
        });
      }
      setExpanded(!expanded);
    }

    function handleClick(event: React.MouseEvent) {
      event.stopPropagation();
      if (!disabled) {
        onTreeSelect?.(event, { id, label, value });
        onNodeSelect?.(event, { id, label, value });
        rest?.onClick?.(event as React.MouseEvent<HTMLElement>);
      }
    }
    function handleKeyDown(event: React.KeyboardEvent<HTMLElement>) {
      function getFocusableNode(node) {
        if (node?.classList.contains(`${prefix}--tree-node`)) {
          return node;
        }
        return node?.firstChild;
      }

      if (disabled) {
        return;
      }

      if (matches(event, [keys.ArrowLeft, keys.ArrowRight, keys.Enter])) {
        event.stopPropagation();
      }

      if (match(event, keys.ArrowLeft)) {
        const findParentTreeNode = (node: Element | null): Element | null => {
          if (!node) return null;
          if (node.classList.contains(`${prefix}--tree-parent-node`)) {
            return node;
          }
          if (node.classList.contains(`${prefix}--tree-node-link-parent`)) {
            return node.firstElementChild;
          }
          if (node.classList.contains(`${prefix}--tree`)) {
            return null;
          }
          return findParentTreeNode(node.parentElement);
        };

        if (children && expanded) {
          if (!enableTreeviewControllable) {
            (onToggle as UncontrolledOnToggle)?.(event, {
              id,
              isExpanded: false,
              label,
              value,
            });
          }
          setExpanded(false);
        } else {
          /**
           * When focus is on a leaf node or a closed parent node, move focus to
           * its parent node (unless its depth is level 1)
           */
          const parentNode = findParentTreeNode(
            (href
              ? currentNode.current?.parentElement?.parentElement
              : currentNode.current?.parentElement) ?? null
          );
          if (parentNode instanceof HTMLElement) {
            parentNode.focus();
          }
        }
      }

      if (children && match(event, keys.ArrowRight)) {
        if (expanded) {
          /**
           * When focus is on an expanded parent node, move focus to the first
           * child node
           */
          getFocusableNode(
            href
              ? currentNode.current?.parentElement?.lastChild?.firstChild
              : currentNode.current?.lastChild?.firstChild
          )?.focus();
        } else {
          if (!enableTreeviewControllable) {
            (onToggle as UncontrolledOnToggle)?.(event, {
              id,
              isExpanded: true,
              label,
              value,
            });
          }
          setExpanded(true);
        }
      }

      if (matches(event, [keys.Enter, keys.Space])) {
        event.preventDefault();
        if (match(event, keys.Enter) && children) {
          // Toggle expansion state for parent nodes
          if (!enableTreeviewControllable) {
            (onToggle as UncontrolledOnToggle)?.(event, {
              id,
              isExpanded: !expanded,
              label,
              value,
            });
          }
          setExpanded(!expanded);
        }
        if (href) {
          currentNode.current?.click();
        }
        handleClick(event as unknown as MouseEvent);
      }
      rest?.onKeyDown?.(event);
    }
    function handleFocusEvent(event: FocusEvent<HTMLElement>) {
      if (event.type === 'focus') {
        rest?.onFocus?.(event);
      }
      if (event.type === 'blur') {
        rest?.onBlur?.(event);
      }
      onNodeFocusEvent?.(event);
    }

    useEffect(() => {
      /**
       * Negative margin shifts node to align with the left side boundary of the
       * tree
       * Dynamically calculate padding to recreate tree node indentation
       * - parent nodes with icon have (depth + 1rem + depth * 0.5) left padding
       * - parent nodes have (depth + 1rem) left padding
       * - leaf nodes have (depth + 2.5rem) left padding without icons (because
       *   of expand icon + spacing)
       * - leaf nodes have (depth + 2rem + depth * 0.5) left padding with icons (because of
       *   reduced spacing between the expand icon and the node icon + label)
       */
      const calcOffset = () => {
        // parent node with icon
        if (children && Icon) {
          return depth + 1 + depth * 0.5;
        }
        // parent node without icon
        if (children) {
          return depth + 1;
        }
        // leaf node with icon
        if (Icon) {
          return depth + 2 + depth * 0.5;
        }
        // leaf node without icon
        return depth + 2.5;
      };

      if (currentNodeLabel.current) {
        currentNodeLabel.current.style.marginInlineStart = `-${calcOffset()}rem`;
        currentNodeLabel.current.style.paddingInlineStart = `${calcOffset()}rem`;
      }

      if (!enableTreeviewControllable) {
        // sync props and state
        setExpanded(isExpanded ?? false);
      }
    }, [
      children,
      depth,
      Icon,
      isExpanded,
      enableTreeviewControllable,
      setExpanded,
    ]);

    const tabIndex = disabled ? undefined : (rest.tabIndex ?? -1);

    const treeNodeProps: React.LiHTMLAttributes<HTMLElement> = {
      ...rest,
      ['aria-current']: !href
        ? isActive || undefined
        : isActive
          ? 'page'
          : undefined,
      ['aria-selected']: !href
        ? disabled
          ? undefined
          : isSelected
        : undefined,
      ['aria-disabled']: disabled,
      ['aria-owns']: children ? `${id}-subtree` : undefined,
      className: treeNodeClasses,
      id,
      onClick: handleClick,
      onKeyDown: handleKeyDown,
      role: 'treeitem',
      tabIndex,
      onFocus: handleFocusEvent,
      onBlur: handleFocusEvent,
    };

    const nodeContent = (
      <div className={`${prefix}--tree-node__label`} ref={currentNodeLabel}>
        {children && (
          // eslint-disable-next-line  jsx-a11y/no-static-element-interactions , jsx-a11y/click-events-have-key-events -- https://github.com/carbon-design-system/carbon/issues/20452
          <span
            className={`${prefix}--tree-parent-node__toggle`}
            onClick={handleToggleClick}>
            <CaretDown className={toggleClasses} />
          </span>
        )}

        <span className={`${prefix}--tree-node__label__details`}>
          {/* eslint-disable-next-line @typescript-eslint/ban-ts-comment -- https://github.com/carbon-design-system/carbon/issues/20452*/}
          {/*@ts-ignore - TS cannot be sure `className` exists on Icon props */}
          {Icon && <Icon className={`${prefix}--tree-node__icon`} />}
          {renderLabelText()}
        </span>
      </div>
    );

    if (href) {
      return (
        <li
          role="none"
          className={children ? `${prefix}--tree-node-link-parent` : ''}>
          <a
            {...treeNodeProps}
            aria-expanded={!!expanded}
            ref={setRefs}
            href={!disabled ? href : undefined}>
            {nodeContent}
          </a>
          {children && (
            <ul
              id={`${id}-subtree`}
              role="group"
              aria-labelledby={nodeLabelId}
              className={classNames(`${prefix}--tree-node__children`, {
                [`${prefix}--tree-node--hidden`]: !expanded,
              })}>
              <DepthContext.Provider value={depth + 1}>
                {children}
              </DepthContext.Provider>
            </ul>
          )}
        </li>
      );
    }

    return (
      <>
        {/* eslint-disable-next-line jsx-a11y/role-supports-aria-props -- https://github.com/carbon-design-system/carbon/issues/20452 */}
        <li
          {...treeNodeProps}
          aria-expanded={children ? !!expanded : undefined}
          ref={setRefs}>
          {nodeContent}
          {children && (
            <ul
              id={`${id}-subtree`}
              role="group"
              aria-labelledby={nodeLabelId}
              className={classNames(`${prefix}--tree-node__children`, {
                [`${prefix}--tree-node--hidden`]: !expanded,
              })}>
              <DepthContext.Provider value={depth + 1}>
                {children}
              </DepthContext.Provider>
            </ul>
          )}
        </li>
      </>
    );
  }
);

TreeNode.propTypes = {
  /**
   * **Note:** this is controlled by the parent TreeView component, do not set manually.
   * The ID of the active node in the tree
   */
  active: deprecate(
    PropTypes.oneOfType([PropTypes.string, PropTypes.number]),
    'The `active` prop for `TreeNode` is no longer needed and has ' +
      'been deprecated. It will be removed in the next major release.'
  ),

  /**
   * Specify the children of the TreeNode
   */
  children: PropTypes.node,

  /**
   * Specify an optional className to be applied to the TreeNode
   */
  className: PropTypes.string,

  /**
   * **[Experimental]** The default expansion state of the node.
   * *This is only supported with the `enable-treeview-controllable` feature flag!*
   */
  defaultIsExpanded: PropTypes.bool,

  /**
   * **Note:** this is controlled by the parent TreeView component, do not set manually.
   * TreeNode depth to determine spacing
   */
  depth: deprecate(
    PropTypes.number,
    'The `depth` prop for `TreeNode` is no longer needed and has ' +
      'been deprecated. It will be removed in the next major release.'
  ),
  /**
   * Specify if the TreeNode is disabled
   */
  disabled: PropTypes.bool,

  /**
   * Specify the TreeNode's ID. Must be unique in the DOM and is used for props.active, props.selected and aria-owns
   */
  id: PropTypes.string,

  /**
   * Specify if the TreeNode is expanded (only applicable to parent nodes)
   */
  isExpanded: PropTypes.bool,

  /**
   * Rendered label for the TreeNode
   */
  label: PropTypes.node,

  /**
   * Callback function for when the node receives or loses focus
   */
  onNodeFocusEvent: deprecate(
    PropTypes.func,
    'The `onNodeFocusEvent` prop for `TreeNode` is no longer needed and has ' +
      'been deprecated. It will be removed in the next major release.'
  ),

  /**
   * Callback function for when the node is selected
   */
  onSelect: PropTypes.func,

  /**
   * Callback function for when a parent node is expanded or collapsed
   */
  onToggle: PropTypes.func,

  /**
   * Callback function for when any node in the tree is selected
   */
  onTreeSelect: deprecate(
    PropTypes.func,
    'The `onTreeSelect` prop for `TreeNode` is no longer needed and has ' +
      'been deprecated. It will be removed in the next major release.'
  ),

  /**
   * A component used to render an icon.
   */
  // eslint-disable-next-line @typescript-eslint/ban-ts-comment -- https://github.com/carbon-design-system/carbon/issues/20452
  // @ts-ignore
  renderIcon: PropTypes.oneOfType([PropTypes.func, PropTypes.object]),

  /**
   * **Note:** this is controlled by the parent TreeView component, do not set manually.
   * Array containing all selected node IDs in the tree
   */
  // eslint-disable-next-line @typescript-eslint/ban-ts-comment -- https://github.com/carbon-design-system/carbon/issues/20452
  // @ts-ignore
  selected: deprecate(
    PropTypes.arrayOf(
      PropTypes.oneOfType([PropTypes.string, PropTypes.number])
    ),
    'The `selected` prop for `TreeNode` is no longer needed and has ' +
      'been deprecated. It will be removed in the next major release.'
  ),

  /**
   * Specify the value of the TreeNode
   */
  value: PropTypes.string,

  /**
   * Optional: The URL the TreeNode is linking to
   */
  href: PropTypes.string,

  /**
   * Specify how the tooltip should align when text is truncated
   */
  align: PropTypes.oneOf([
    'top',
    'bottom',
    'left',
    'right',
    'top-start',
    'top-end',
    'bottom-start',
    'bottom-end',
    'left-end',
    'left-start',
    'right-end',
    'right-start',
  ]),

  /**
   * **Experimental**: Will attempt to automatically align the floating
   * element to avoid collisions with the viewport and being clipped by
   * ancestor elements. Requires React v17+
   * @see https://github.com/carbon-design-system/carbon/issues/18714
   */
  autoAlign: PropTypes.bool,
};

TreeNode.displayName = 'TreeNode';
export default TreeNode;
