/**
 * Copyright IBM Corp. 2016, 2026
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import PropTypes from 'prop-types';
import React, {
  Children,
  forwardRef,
  useContext,
  type ElementType,
  type HTMLAttributes,
  type ReactElement,
  type ReactNode,
} from 'react';
import {
  PolymorphicComponentPropWithRef,
  PolymorphicRef,
} from '../../internal/PolymorphicProps';
import { TextDirectionContext, type TextDir } from './TextDirectionContext';

export interface TextBaseProps {
  dir?: TextDir;
  children?: ReactNode;
}

export type TextProps<T extends ElementType> = PolymorphicComponentPropWithRef<
  T,
  TextBaseProps
>;

type TextComponent = <T extends ElementType = 'span'>(
  props: TextProps<T> & { ref?: PolymorphicRef<T> }
) => ReactElement | null;

// eslint-disable-next-line react/display-name -- https://github.com/carbon-design-system/carbon/issues/20452
export const Text = forwardRef<
  HTMLSpanElement,
  TextBaseProps & {
    as?: ElementType;
    dir?: 'auto' | 'ltr' | 'rtl';
  } & HTMLAttributes<HTMLSpanElement>
>(({ as, children, dir = 'auto', ...rest }, ref) => {
  const context = useContext(TextDirectionContext);
  const textProps: { dir?: TextDir } = {};
  const BaseComponent = as ?? 'span';
  const value = {
    ...context,
  };

  if (!context) {
    textProps.dir = dir;
    value.direction = dir;
  } else {
    const { direction: parentDirection, getTextDirection } = context;

    if (getTextDirection && getTextDirection.current) {
      const text = getTextFromChildren(children);
      const override = getTextDirection.current(text);

      if (parentDirection !== override) {
        textProps.dir = override;
        value.direction = override;
      } else if (parentDirection === 'auto') {
        textProps.dir = override;
      }
    } else if (parentDirection !== dir) {
      textProps.dir = dir;
      value.direction = dir;
    } else if (parentDirection === 'auto') {
      textProps.dir = dir;
    }
  }

  return (
    <TextDirectionContext.Provider value={value}>
      <BaseComponent ref={ref} {...rest} {...textProps}>
        {children}
      </BaseComponent>
    </TextDirectionContext.Provider>
  );
}) as TextComponent;

// @ts-expect-error - `propTypes` isn't typed.
Text.propTypes = {
  /**
   * Provide a custom element type used to render the outermost node
   */
  as: PropTypes.oneOfType([
    PropTypes.func,
    PropTypes.string,
    PropTypes.elementType,
  ]),

  /**
   * Provide child elements or text to be rendered inside of this component
   */
  children: PropTypes.node,

  /**
   * Specify the text direction to be used for this component and any of its
   * children
   */
  dir: PropTypes.oneOf(['ltr', 'rtl', 'auto']),
};

const getTextFromChildren = (children: ReactNode) => {
  if (typeof children === 'string') {
    return children;
  }

  const text = Children.map(children, (child) => {
    if (typeof child === 'string') {
      return child;
    }
    return null;
  })?.filter((text) => {
    return text !== null;
  });

  if (text?.length === 1) {
    return text[0];
  }

  return text;
};
