/**
 * Copyright IBM Corp. 2016, 2026
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import userEvent from '@testing-library/user-event';
import '@testing-library/jest-dom/jest-globals';
import { expect } from '@jest/globals';
import { act } from 'react';

const prefix = 'cds';

// Finding nodes in a ListBox
export const findListBoxNode = () => {
  return document.querySelector('.cds--list-box');
};

export const findMenuNode = () => {
  return document.querySelector(`.${prefix}--list-box__menu`);
};

export const findMenuItemNode = (index: number) => {
  const nodes = document.querySelectorAll(`.${prefix}--list-box__menu-item`);
  const node = nodes.item(index);

  if (node) {
    return node;
  }

  throw new Error(`Unable to find node at index: ${index} in: ${nodes}`);
};

export const findMenuIconNode = () => {
  return document.querySelector(`.${prefix}--list-box__menu-icon`);
};

export const findFieldNode = () => {
  return document.querySelector(`.${prefix}--list-box__field`);
};

export const findComboboxNode = () => {
  return document.querySelector(`.${prefix}--list-box[role="combobox"]`);
};

export const findPopupNode = () => {
  return document.querySelector('[aria-haspopup="listbox"]');
};

// Actions
export const openMenu = async () => {
  const field = findFieldNode();

  if (!field) {
    throw new Error('Unable to find field node.');
  }

  await userEvent.click(field);
};

// Common assertions, useful for validating a11y props are set when needed
export const assertMenuOpen = (mockProps: { items: unknown[] }) => {
  const menu = findMenuNode();
  const icon = findMenuIconNode();
  const popup = findPopupNode();

  if (!menu) throw new Error('Unable to find menu node.');
  if (!icon) throw new Error('Unable to find menu icon node.');
  if (!popup) throw new Error('Unable to find popup node.');

  expect(menu.childNodes.length).toBe(mockProps.items.length);

  expect(icon).toHaveClass(`${prefix}--list-box__menu-icon--open`);
  expect(popup).toHaveAttribute('aria-expanded', 'true');
};

export const assertMenuClosed = () => {
  const icon = findMenuIconNode();
  const popup = findPopupNode();

  if (!icon) throw new Error('Unable to find menu icon node.');
  if (!popup) throw new Error('Unable to find popup node.');

  expect(icon).toHaveClass(`${prefix}--list-box__menu-icon`);
  expect(icon).not.toHaveClass(`${prefix}--list-box__menu-icon--open`);
  expect(popup).toHaveAttribute('aria-expanded', 'false');
};

/**
 * 'GenericItem' corresponds to an item in a collection that is passed to
 * `MultiSelect` that is in a predictable shape and works with the default
 * `itemToString` function.
 */
export const generateGenericItem = (index: number) => ({
  id: `id-${index}`,
  label: `Item ${index}`,
  value: index,
});

/**
 * 'CustomItem' corresponds to a potentially different item structure that might
 * be passed into `MultiSelect` that we would need to supply a custom
 * `itemToString` method for.
 */
export const generateCustomItem = (index: number) => ({
  field: `Item ${index}`,
  value: `Custom value ${index}`,
});

/**
 * Generates an array of values generated by the `generator` function.
 */
export const generateItems = <T>(
  amount: number,
  generator: (index: number) => T
) => Array.from({ length: amount }, (_, i) => generator(i));

export const customItemToString = ({
  field,
}: {
  field: string;
  value: string;
}) => field;

/**
 * This object contains two sets of three items that share the same root
 * word in different portions of the string (beginning, middle, end):
 *
 *  - 'struct'
 *  - 'port'
 *
 * Separated by a disabled item, these derivative words are helpful when
 * testing fuzzy search functions and components that do substring filtering.
 */
export const cognateItems = [
  {
    id: 'structure',
    text: 'Structure',
  },
  {
    id: 'construct',
    text: 'Construct',
  },
  {
    id: 'instruction',
    text: 'Instruction',
  },
  {
    id: 'disabled-item',
    text: 'A disabled item',
    disabled: true,
  },
  {
    id: 'transport',
    text: 'Transport',
  },
  {
    id: 'portable',
    text: 'Portable',
  },
  {
    id: 'import',
    text: 'Import',
  },
];

/**
 * Flushes microtasks to ensure element position state is settled
 * From https://floating-ui.com/docs/react#testing
 * More context here: https://github.com/floating-ui/react-popper/issues/368#issuecomment-1340413010
 */
export const waitForPosition = () => act(async () => {});
