/**
 * Copyright IBM Corp. 2016, 2025
 *
 * This source code is licensed under the Apache-2.0 license found in the
 * LICENSE file in the root directory of this source tree.
 */

import React from 'react';
import CheckboxGroup from '../CheckboxGroup';
import Checkbox from '../Checkbox/Checkbox';
import { render, screen } from '@testing-library/react';
import { AILabel } from '../AILabel';

const prefix = 'cds';

/**
 * @param element {Element}
 * @returns {Record<string, string>}
 */
const getElementAttributes = ({ attributes }) =>
  Array.from(attributes).reduce(
    (acc, { name, value }) => ({ ...acc, [name]: value }),
    {}
  );

describe('CheckboxGroup', () => {
  it('should support a custom `className` prop on the outermost element', () => {
    const { container } = render(
      <CheckboxGroup className="test" legendText="Checkbox heading" />
    );
    expect(container.firstChild).toHaveClass('test');
  });

  it('should render helperText', () => {
    render(
      <CheckboxGroup
        className="test"
        legendText="Checkbox heading"
        helperText="Helper text"
      />
    );

    expect(screen.getByText('Helper text')).toBeInTheDocument();
  });

  it('should render helperText with value 0', () => {
    render(<CheckboxGroup legendText="heading" helperText={0} />);
    expect(screen.getByText('0')).toBeInTheDocument();
  });

  it('should set data-invalid when invalid prop is true', () => {
    const { container } = render(
      <CheckboxGroup
        className="some-class"
        legendText="Checkbox heading"
        invalid>
        <Checkbox
          defaultChecked
          labelText="Checkbox label"
          id="checkbox-label-1"
        />
        <Checkbox labelText={`Checkbox label`} id="checkbox-label-2" />
      </CheckboxGroup>
    );

    expect(container.firstChild).toHaveAttribute('data-invalid', 'true');
  });

  it('should display invalidText if invalid prop is true', () => {
    render(
      <CheckboxGroup
        className="some-class"
        legendText="Checkbox heading"
        invalid
        invalidText="Invalid text">
        <Checkbox
          defaultChecked
          labelText="Checkbox label"
          id="checkbox-label-1"
        />
        <Checkbox labelText={`Checkbox label`} id="checkbox-label-2" />
      </CheckboxGroup>
    );

    expect(screen.getByText('Invalid text')).toBeInTheDocument();
  });

  it('should render legendText', () => {
    render(<CheckboxGroup className="test" legendText="Checkbox heading" />);

    expect(screen.getByText('Checkbox heading')).toBeInTheDocument();
  });

  it('should set the id for legend based on legendId', () => {
    render(<CheckboxGroup legendId="legend-testid" legendText="legendtest" />);

    expect(screen.getByText('legendtest')).toHaveAttribute(
      'id',
      'legend-testid'
    );
  });

  it('should respect readOnly prop', () => {
    const { container } = render(
      <CheckboxGroup
        className="some-class"
        legendText="Checkbox heading"
        readOnly>
        <Checkbox
          defaultChecked
          labelText="Checkbox label"
          id="checkbox-label-1"
        />
        <Checkbox labelText={`Checkbox label`} id="checkbox-label-2" />
      </CheckboxGroup>
    );

    expect(container.firstChild).toHaveClass(
      `${prefix}--checkbox-group--readonly`
    );
  });

  it('should respect warn prop', () => {
    const { container } = render(
      <CheckboxGroup className="some-class" legendText="Checkbox heading" warn>
        <Checkbox
          defaultChecked
          labelText="Checkbox label"
          id="checkbox-label-1"
        />
        <Checkbox labelText={`Checkbox label`} id="checkbox-label-2" />
      </CheckboxGroup>
    );

    // eslint-disable-next-line testing-library/no-container, testing-library/no-node-access
    const warnIcon = container.querySelector(
      `svg.${prefix}--checkbox__invalid-icon--warning`
    );

    expect(container.firstChild).toHaveClass(
      `${prefix}--checkbox-group--warning`
    );
    expect(warnIcon).toBeInTheDocument();
  });

  it('should display warnText if warn prop is true', () => {
    render(
      <CheckboxGroup
        className="some-class"
        legendText="Checkbox heading"
        warn
        warnText="Warn text">
        <Checkbox
          defaultChecked
          labelText="Checkbox label"
          id="checkbox-label-1"
        />
        <Checkbox labelText={`Checkbox label`} id="checkbox-label-2" />
      </CheckboxGroup>
    );

    expect(screen.getByText('Warn text')).toBeInTheDocument();
    expect(screen.getByText('Warn text')).toHaveClass(
      `${prefix}--form-requirement`
    );
  });

  it('should respect deprecated slug prop', () => {
    const spy = jest.spyOn(console, 'warn').mockImplementation(() => {});
    const { container } = render(
      <CheckboxGroup
        className="some-class"
        legendText="Checkbox heading"
        slug={<AILabel />}
      />
    );

    expect(container.firstChild).toHaveClass(`${prefix}--checkbox-group--slug`);
    spy.mockRestore();
  });

  it('should respect decorator prop', () => {
    const { container } = render(
      <CheckboxGroup
        className="some-class"
        legendText="Checkbox heading"
        decorator={<AILabel />}
      />
    );

    expect(container.firstChild).toHaveClass(
      `${prefix}--checkbox-group--decorator`
    );
  });

  it('should render checkboxes horizontally', () => {
    const { container } = render(
      <CheckboxGroup orientation="horizontal" legendText="test-horizontal-prop">
        <Checkbox labelText="Checkbox label 1" id="checkbox-label-1" />
        <Checkbox labelText="Checkbox label 2" id="checkbox-label-2" />
        <Checkbox labelText="Checkbox label 3" id="checkbox-label-3" />
      </CheckboxGroup>
    );

    expect(container.firstChild).toHaveClass(
      `${prefix}--checkbox-group--horizontal`
    );
  });

  describe('prop inheritance', () => {
    it('should pass props to child `Checkbox` components', () => {
      render(
        <CheckboxGroup legendText="Checkbox heading" invalid readOnly warn>
          <Checkbox labelText="Checkbox 1" id="checkbox-1" />
          <Checkbox labelText="Checkbox 2" id="checkbox-2" />
        </CheckboxGroup>
      );

      const checkbox1 = screen.getByLabelText('Checkbox 1');
      const checkbox2 = screen.getByLabelText('Checkbox 2');
      const attributes1 = getElementAttributes(checkbox1);
      const attributes2 = getElementAttributes(checkbox2);

      expect(attributes1).toEqual({
        class: `${prefix}--checkbox`,
        id: 'checkbox-1',
        'data-invalid': 'true',
        'aria-readonly': 'true',
        type: 'checkbox',
      });
      expect(attributes2).toEqual({
        class: `${prefix}--checkbox`,
        id: 'checkbox-2',
        'data-invalid': 'true',
        'aria-readonly': 'true',
        type: 'checkbox',
      });
    });

    it('should not override individual `Checkbox` props', () => {
      render(
        <CheckboxGroup legendText="Checkbox heading" readOnly>
          <Checkbox labelText="Checkbox 1" id="checkbox-1" readOnly={false} />
          <Checkbox labelText="Checkbox 2" id="checkbox-2" />
        </CheckboxGroup>
      );

      const checkbox1 = screen.getByLabelText('Checkbox 1');
      const checkbox2 = screen.getByLabelText('Checkbox 2');
      const attributes1 = getElementAttributes(checkbox1);
      const attributes2 = getElementAttributes(checkbox2);

      expect(attributes1).toEqual({
        class: `${prefix}--checkbox`,
        id: 'checkbox-1',
        // Should not be read-only because it explicitly sets `readOnly` to
        // `false`.
        'aria-readonly': 'false',
        type: 'checkbox',
      });
      expect(attributes2).toEqual({
        class: `${prefix}--checkbox`,
        id: 'checkbox-2',
        // Should be read-only because it inherits from the group.
        'aria-readonly': 'true',
        type: 'checkbox',
      });
    });

    it('should not affect non-`Checkbox` children', () => {
      render(
        <CheckboxGroup legendText="Checkbox heading" readOnly>
          <Checkbox labelText="Checkbox label" id="checkbox-1" />
          <div data-testid="non-checkbox">Not a checkbox</div>
        </CheckboxGroup>
      );

      const checkbox = screen.getByLabelText('Checkbox label');
      const nonCheckbox = screen.getByTestId('non-checkbox');
      const checkboxAttributes = getElementAttributes(checkbox);
      const nonCheckboxAttributes = getElementAttributes(nonCheckbox);

      expect(checkboxAttributes).toEqual({
        class: `${prefix}--checkbox`,
        id: 'checkbox-1',
        'aria-readonly': 'true',
        type: 'checkbox',
      });
      expect(nonCheckboxAttributes).toEqual({
        'data-testid': 'non-checkbox',
      });
    });
  });
});
