import PropTypes from 'prop-types';
import React, { useCallback } from 'react';
import { Pagination as CarbonPagination } from '@carbon/react';

/**
 * Wrapped version of Carbon `<Pagination>`, that uses zero-based starting row
 * index instead of page number.
 */
const Pagination = ({
  start,
  count,
  pageSize,
  pageSizes,
  onChangeStart,
  onChangePageSize,
}) => {
  const handleChangePage = useCallback(
    ({ page: newPage, pageSize: newPageSize }) => {
      if (onChangePageSize && pageSize !== newPageSize) {
        onChangePageSize({ pageSize: newPageSize });
      }
      const page = Math.floor(start / pageSize) + 1;
      if (page !== newPage) {
        const newStart = Math.min(
          Math.max(start + (newPage - page) * pageSize, 0),
          count
        );
        if (onChangeStart && start !== newStart) {
          onChangeStart({ start: newStart });
        }
      }
    },
    [start, count, pageSize, onChangeStart, onChangePageSize]
  );

  return (
    <CarbonPagination
      page={Math.floor(start / pageSize) + 1}
      pageSize={pageSize}
      pageSizes={pageSizes}
      totalItems={count}
      onChange={handleChangePage}
    />
  );
};

Pagination.propTypes = {
  /**
   * Total number of rows.
   */
  count: PropTypes.number.isRequired,

  /**
   * Callback function for page size change
   */
  onChangePageSize: PropTypes.func,

  /**
   * Callback function for page start change
   */
  onChangeStart: PropTypes.func,

  /**
   * Number of items per page.
   */
  pageSize: PropTypes.number.isRequired,

  /**
   * List of page sizes.
   */
  pageSizes: PropTypes.arrayOf(PropTypes.number).isRequired,

  /**
   * The row number where current page start with, index that starts with zero. Corresponds to the attribute with the same name.
   */
  start: PropTypes.number.isRequired,
};

export default Pagination;
